//+------------------------------------------------------------------+
//|                                                  P4L BoxFibo.mq4 |
//|                               A significant rewrite by pips4life |
//|                 Original BoxFibo.mq4 by hanover + mods by others |
//+------------------------------------------------------------------+
#property copyright "v2.00+ by pips4life"
#property link      "https://www.forexfactory.com/thread/662529-modified-useful-mq4-utilities-indicators-and-related-tips"
#property indicator_chart_window
#property strict

#property description "This indicator draws boxes around *completed* time-windows from: StartTime(inclusive) to EndTime(exclusive)"
#property description "ICT Standard Deviation lines (20 above and 20 below) are drawn based on the box price range"
#property description "Horizontal lines are drawn at top, middle and bottom of box with customizable time extension"
#property description "Standard Deviation lines extend from the box until 23:00 of the same day"
#property description "On/Off button toggles the display.  The indicator is intended for Timeframes <= H1."
#property description "One may add more-than-one of this indicator so long as either the StartTime and/or EndTime differs."

#property version "2.01"
extern string _VERSION_v2_01___2023_Apr_26 = "By pips4life. (Orig: hanover)";

/*
VERSION HISTORY:

2023-Apr-26 v2_01 by pips4life
   Removed Fibonacci levels
   Added horizontal lines at top, middle and bottom of box with customizable time extension
   Increased standard deviation lines to 20 above and 20 below
   Standard deviation lines now extend from box until 23:00 of the same day
   Added numbers (1-20) on standard deviation lines with customizable position, size and color
   Added option to show/hide middle horizontal line
   Added option to show/hide standard deviation lines
   Added customizable line width and style for standard deviation lines
   Only the last box numbers are flexible horizontally but stay fixed vertically with their lines
   Added text on horizontal lines with customizable position, size and color
   Removed AutoFlipColorsIfDarkBG option
   Fixed numbers to stay visible when scrolling and work on weekends
   Changed internal 'debugmicro' value from "TRUE" to "false", as was intended for a production release.
   
2023-Apr-25 v2_00 by pips4life
   NOTE: These are major changes, and it would not be unusual if there are still bugs!  Please report any you find.
   Optimized performance.   Previous version ran *every* tick, and depending up TF, might take 400 - 1000 microseconds (which is a problem during high-tick news events!)
     The previous version used very slow (wasteful) functions StrToTime and TimeToStr far too often. Replaced with faster equivalent calculations. 
     The previous version also recalculated ALL the previous boxes, for every tick, which is a total waste of CPU!  
     This version runs only once-per-second; OR, if a new bar; OR, if old bars have refreshed.  It does not recalculate old boxes unnecessarily.  Most ticks are 1-5 microseconds!
     (This version is not completely optimized to prevent ALL unnecessary re-calculations, but it's far better than before and good enough.)
   Because this version is much faster, the NumDays default is now 30 instead of 10 (Users may customize as desired).
   Added on/off button.
   Eliminated 'identity' variable, and replaced with internal *unique* 'createdObjectPrefix' (so long as either StartTime or EndTime are unique!)
   The default BoxColor & FiboColor are best for lightBG charts.
   New var 'SupressIndicatorForTFsAbove_minutes' (60) will suppress the BoxFibo's for TF's > 60 minutes (H1). (This indicator is really only 
      intended for <= H1 timeframes anyway). Note: Toggling the button will *ignore* this variable, and force-display on <= H4 timeframes.
   The previous version suffered from a few major bugs which might be infrequent, but would nevertheless occur:  
      * New boxes would overdraw old boxes, which would make them completely invisible!
      * The StartTime might include the previous bar on certain Symbols (e.g. XAUUSD (Gold) often does not have a bar at 00:00. The old version included the "23.00" bar.

2009-2023 Various mods by others.

2009-Jun-07  Orig. by hanover. https://www.forexfactory.com/thread/post/2785607#post2785607

*/

string createdObjectPrefix = "_BoxFibo_"; // This var (replaces 'identity' is modified in OnInit(), and will be unique so long as StartTime & EndTime are unique.

extern string StartTime                           = "00:00"; 
extern string EndTime                             = "09:00"; 
extern int    NumDays                             = 30;
extern color  BoxColor                            = clrLightBlue;

// Standard Deviation Lines Settings
extern bool   ShowStandardDeviationLines          = true;          // Show standard deviation lines
extern color  ICT_SD_Color                        = clrRed;        // Color for ICT Standard Deviation lines
extern int    ICT_SD_LineWidth                    = 1;             // Width for ICT Standard Deviation lines
extern ENUM_LINE_STYLE ICT_SD_LineStyle           = STYLE_DASH;    // Style for ICT Standard Deviation lines

// NEW: Option to hide specific standard deviation levels
extern string HideStandardDeviationLevels         = "1,3,5,7,9,11,13,15,17,19"; // Comma-separated list of levels to hide (1-20)

// Standard Deviation Numbers Settings
extern bool   ShowNumbersOnSDLines                = true;          // Show numbers on standard deviation lines
extern color  NumbersTextColor                    = clrRed;        // Color for numbers text
extern int    NumbersTextSize                     = 8;             // Size for numbers text
extern int    NumbersTextPosition                 = 0;             // 0=Start, 1=Middle, 2=End of line

// Horizontal Lines Settings
extern string HorizontalLinesStartTime            = "00:00";       // Start time for horizontal lines
extern string HorizontalLinesEndTime              = "23:00";       // End time for horizontal lines
extern color  HorizontalLinesColor                = clrBlue;       // Color for horizontal lines
extern int    HorizontalLinesWidth                = 2;             // Width for horizontal lines
extern bool   ShowMiddleLine                      = true;          // Show middle horizontal line

// Horizontal Lines Text Settings
extern bool   ShowTextOnHorizontalLines           = true;          // Show text on horizontal lines
extern string HorizontalLinesTopText              = "High";        // Text for top horizontal line
extern string HorizontalLinesMiddleText           = "Mid";         // Text for middle horizontal line
extern string HorizontalLinesBottomText           = "Low";         // Text for bottom horizontal line
extern color  HorizontalLinesTextColor            = clrBlue;       // Color for horizontal lines text
extern int    HorizontalLinesTextSize             = 8;             // Size for horizontal lines text
extern int    HorizontalLinesTextPosition         = 2;             // 0=Start, 1=Middle, 2=End of line

extern int    SupressIndicatorForTFsAbove_minutes = 60;

//Forex-Station button template start41; copy and paste
extern string             button_note1_          = "------------------------------";
extern int                btn_Subwindow          = 0;                               // What window to put the button on.  If <0, the button will use the same sub-window as the indicator.
extern ENUM_BASE_CORNER   btn_corner             = CORNER_LEFT_UPPER;               // button corner on chart for anchoring
extern string             btn_text               = "BoxFibo";                       // a button name
extern string             btn_Font               = "Arial";                         // button font name
extern int                btn_FontSize           = 9;                               // button font size               
extern color              btn_text_ON_color      = clrLime;                         // ON color when the button is turned on
extern color              btn_text_OFF_color     = clrLightSalmon;                  // OFF color when the button is turned off
extern color              btn_background_color   = clrDimGray;                      // background color of the button
extern color              btn_border_color       = clrBlack;                        // border color the button
extern int                button_x               = 20;                              // x coordinate of the button     
extern int                button_y               = 45;                              // y coordinate of the button     
extern int                btn_Width              = 60;                              // button width
extern int                btn_Height             = 20;                              // button height
extern bool               btn_Initial_Default_ON = false;                           // When first added to chart, false enables OFF status of button by default
extern string             button_note2           = "------------------------------";
string buttonId;
bool buttonStatus;

// Array to store levels to hide
int hiddenLevels[];
bool levelsParsed = false;

datetime dtStartTime;
datetime dtEndTime;
datetime dtHorizontalLinesStartTime;
datetime dtHorizontalLinesEndTime;
bool debugmicro = false;
datetime lastBoxDateTime = 0;

//+------------------------------------------------------------------+
int OnInit()
  {
   createdObjectPrefix = StringConcatenate(createdObjectPrefix,StartTime,"_",EndTime,"_");
   //===========  Determine if duplicate Indicator exists
   string shortName = createdObjectPrefix; // Must NOT be simply the same as 'WindowExpertName()', else it will see itself as a duplicate.
   int winFind = WindowFind(shortName);
   if(winFind >= 0)
   {  
      Alert("ERROR(?).  The chart may already have this indicator with the same StartTime & EndTime."); // Because of false errors, do not remove.  //This duplicate is auto-removed.");
      //return(INIT_FAILED); // Due to possible false errors, commented out to keep going.
   }  
   IndicatorShortName(shortName);
   //============  End of check, and adjustment, if duplicate Indicator added.
   
   dtStartTime = StrToTime("1970.01.01 "+StartTime);
   dtEndTime = StrToTime("1970.01.01 "+EndTime);
   dtHorizontalLinesStartTime = StrToTime("1970.01.01 "+HorizontalLinesStartTime);
   dtHorizontalLinesEndTime = StrToTime("1970.01.01 "+HorizontalLinesEndTime);
   //Alert("DEBUG dtStartTime=",dtStartTime,"  dtEndTime=",dtEndTime);
   
   // Parse the hidden levels string
   parseHiddenLevels();
   
   // --------------- Button related ----------------- //
   //if(btn_Subwindow<0) btn_Subwindow = swin;
   // The leading "_" gives buttonId a *unique* prefix so it is not deleted if/when any other chart objects are deleted.
   buttonId = "_" + createdObjectPrefix + "_BT_";
   if (ObjectFind(buttonId)<0)
   { 
      createButton(buttonId, btn_text, btn_Width, btn_Height, btn_Font, btn_FontSize, btn_background_color, btn_border_color, btn_text_ON_color);
      if(!btn_Initial_Default_ON) ObjectSetInteger(0,buttonId, OBJPROP_STATE,false); // This can force the initially created button to start as "off".  (But just changing TF's keeps current status!)
   }
   ObjectSetInteger(0, buttonId, OBJPROP_YDISTANCE, button_y);
   ObjectSetInteger(0, buttonId, OBJPROP_XDISTANCE, button_x);
   buttonStatus = ObjectGetInteger(0, buttonId, OBJPROP_STATE);   
   if (buttonStatus) ObjectSetInteger(0,buttonId,OBJPROP_COLOR,btn_text_ON_color); 
   else ObjectSetInteger(0,buttonId,OBJPROP_COLOR,btn_text_OFF_color);
   // --------------- -------------- ----------------- //
   
   // REMOVED: Automatic plotting in OnInit - indicator will only plot when button is pressed
   // if(Period() <= SupressIndicatorForTFsAbove_minutes) plot_obj();
   return(INIT_SUCCEEDED);
}//end of OnInit()

//+------------------------------------------------------------------+
void parseHiddenLevels()
{
   ArrayResize(hiddenLevels, 0);
   string levelsArray[];
   string sep = ",";
   ushort u_sep = StringGetCharacter(sep,0);
   
   int count = StringSplit(HideStandardDeviationLevels, u_sep, levelsArray);
   
   for(int i = 0; i < count; i++)
   {
      string levelStr = StringTrimLeft(StringTrimRight(levelsArray[i]));
      int level = (int)StringToInteger(levelStr);
      if(level >= 1 && level <= 20)
      {
         int size = ArraySize(hiddenLevels);
         ArrayResize(hiddenLevels, size + 1);
         hiddenLevels[size] = level;
      }
   }
   levelsParsed = true;
}

//+------------------------------------------------------------------+
bool isLevelHidden(int level)
{
   if(!levelsParsed) parseHiddenLevels();
   
   for(int i = 0; i < ArraySize(hiddenLevels); i++)
   {
      if(hiddenLevels[i] == level)
         return true;
   }
   return false;
}

//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   // If just changing a TF', the button need not be deleted, therefore the 'OBJPROP_STATE' is also preserved.
   if ( reason != REASON_CHARTCHANGE
     && reason != REASON_CLOSE
     // && reason != REASON_RECOMPILE  // Leave commented, if personal preference is to delete it when Re-Compiled.
      ) ObjectDelete(buttonId);

   IndicatorShortName("any_random_name_just_before_deinit_is_done");
   del_obj();
   return;
}//end of OnDeinit()

//+------------------------------------------------------------------------------------------------------------------+
void OnChartEvent(const int id, //don't change anything here
                  const long &lparam,
                  const double &dparam,
                  const string &sparam)
{
   
   // Method 2 (Preferred):  Check for a specific id and a specific sparam (object name) of interest, and ONLY act on that event!
   if (id==CHARTEVENT_OBJECT_CLICK && sparam == buttonId)
   {
      buttonStatus = ObjectGetInteger(0, buttonId, OBJPROP_STATE);
      
      if (buttonStatus) // on
      {
         ObjectSetInteger(0,buttonId,OBJPROP_COLOR,btn_text_ON_color); 
         //Call a function (preferably not the entire start() function though it might work) to do whatever the indicator does here:
         del_obj();
         if(Period() < PERIOD_D1) plot_obj(); // The button *ignores* 'SupressIndicatorForTFsAbove_minutes' and can therefore force H4 (but not >= D1)         
      }
      else // off
      {
         ObjectSetInteger(0,buttonId,OBJPROP_COLOR,btn_text_OFF_color);
         // If this indy had created objects (besides the buttonId), this would be the place to remove them one time only, now that the indy is "off"
         // deleteCreatedObjects(IndicatorObjPrefix);
         del_obj();
      }
      ChartRedraw();
   }
}//end of OnChartEvent()

//+------------------------------------------------------------------------------------------------------------------+
// Normally one does not need to change anything here
void createButton(string buttonID,string buttonText,int width2,int height,string font,int fontSize,color bgColor,color borderColor,color txtColor, bool initial_state=true)
{
      ObjectDelete    (0,buttonID);
      ObjectCreate    (0,buttonID,OBJ_BUTTON,btn_Subwindow,0,0);
      ObjectSetInteger(0,buttonID,OBJPROP_COLOR,txtColor);
      ObjectSetInteger(0,buttonID,OBJPROP_BGCOLOR,bgColor);
      ObjectSetInteger(0,buttonID,OBJPROP_BORDER_COLOR,borderColor);
      ObjectSetInteger(0,buttonID,OBJPROP_BORDER_TYPE,BORDER_RAISED);
      ObjectSetInteger(0,buttonID,OBJPROP_XSIZE,width2);
      ObjectSetInteger(0,buttonID,OBJPROP_YSIZE,height);
      ObjectSetString (0,buttonID,OBJPROP_FONT,font);
      ObjectSetString (0,buttonID,OBJPROP_TEXT,buttonText);
      ObjectSetInteger(0,buttonID,OBJPROP_FONTSIZE,fontSize);
      ObjectSetInteger(0,buttonID,OBJPROP_SELECTABLE,0);
      ObjectSetInteger(0,buttonID,OBJPROP_CORNER,btn_corner);
      ObjectSetInteger(0,buttonID,OBJPROP_HIDDEN,1);
      ObjectSetInteger(0,buttonID,OBJPROP_XDISTANCE,9999);
      ObjectSetInteger(0,buttonID,OBJPROP_YDISTANCE,9999);
      // Upon creation, set the 'initial state'  If "true" which is "on" (the default), one will see the indicator by default
      ObjectSetInteger(0, buttonId, OBJPROP_STATE, initial_state);
}//end of createButton()

//+------------------------------------------------------------------+
int start()  
{  
   // Only run if button is ON
   if(!buttonStatus) return(0);
   
   if(Period() > SupressIndicatorForTFsAbove_minutes) return(0);  // User can control whether to suppress ABOVE some TF, like > H1
   if(Period() >= PERIOD_D1) return(0);  // Indicator does not apply at all to D1 and above (and is not really intended for H4 either)
      
   int limit;
   int counted_bars = IndicatorCounted();
   //---- check for possible errors 
   if(counted_bars<0) return(-1); 
   limit=Bars-counted_bars; 
   
   static datetime last_timecurrent = 0;
   bool isNewBar = IsNewBar(); 
   if(TimeCurrent() == last_timecurrent && !isNewBar && limit<=1 ) return(0);  // Run no more than once per second. 
   last_timecurrent = TimeCurrent();
   
   
   ulong microstart = 0;
   if(debugmicro) microstart = GetMicrosecondCount();
   plot_obj(limit);
//+------------------------------------------------------------------+
   if(debugmicro)
   {
      static int nn=0;
      static int uslimit = 25;
      ulong time_us = GetMicrosecondCount() - microstart;
      if(time_us > 400) 
      {
         if (nn > uslimit) 
         {
            nn=uslimit;
            uslimit = uslimit + 2;
         }
      }
      nn++;
      if (nn<=uslimit || isNewBar) Print("Box Fibo: Pass n=",nn," took: ",time_us," micro-seconds.  limit=",limit,"  Period()=",Period());
   }
  return(0);
}//end of start()

//+------------------------------------------------------------------+
void plot_obj(int limit=INT_MAX)  {
//+------------------------------------------------------------------+
   
   datetime timecurrent = TimeCurrent();
   timecurrent -= (datetime) MathMod(timecurrent,86400);
   datetime dt1 = timecurrent + dtStartTime;
   datetime dt2 = timecurrent + dtEndTime;
   int      dys = 0, NumBars = (int) (dt2-dt1)/PeriodSeconds();
   limit = MathMin(limit,Bars-NumBars);
   
   // Track the most recent box datetime
   datetime currentLastBox = 0;
   
   for (int i=0; i<limit; i++)  
   {
      datetime timei = Time[i] - (datetime) MathMod(Time[i],86400);
      dt1 = timei + dtStartTime;
      dt2 = timei + dtEndTime;
      int ib1 = iBarShift(NULL,0,dt1);
      if(Time[ib1] < dt1 && ib1>=1) { ib1--; dt1 = Time[ib1]; } // If not an EXACT match for the bar time, the *next* bar is correct, so shift 1 bar forward.
      int ib2 = iBarShift(NULL,0,dt2);
      if(Time[ib2] < dt2 && ib2>=1) { ib2--; dt2 = Time[ib2]; }
      if (dt2 != Time[i])  continue;

      int barHigh = iHighest(NULL,0,MODE_HIGH,ib1-ib2,ib2+1);
      int barLow = iLowest(NULL,0,MODE_LOW,ib1-ib2,ib2+1);
      double vHigh = High[barHigh];
      double vLow = Low[barLow];
      
      string objname = createdObjectPrefix+IntegerToString(dt1)+"_R";
      if( ObjectFind(objname) < 0) ObjectCreate(objname,OBJ_RECTANGLE,0,dt1,vHigh,dt2-60,vLow);
      else
      {
         ObjectSetInteger(0,objname,OBJPROP_TIME1,dt1);
         ObjectSetInteger(0,objname,OBJPROP_TIME2,dt2-60);
         ObjectSetDouble(0,objname,OBJPROP_PRICE1,vHigh);
         ObjectSetDouble(0,objname,OBJPROP_PRICE2,vLow);
      }
      ObjectSet(objname,OBJPROP_COLOR,BoxColor);
      ObjectSet(objname,OBJPROP_SELECTABLE,false); 
      
      // Add horizontal lines at top, middle and bottom with customizable time extension
      drawHorizontalLines(dt1, vHigh, vLow);
      
      // Track the most recent box
      if(dt1 > currentLastBox) currentLastBox = dt1;
      
      dys++; if (dys >= NumDays)   break;
   }
   
   // Update the global last box datetime
   if(currentLastBox > 0) lastBoxDateTime = currentLastBox;
   
   // Now draw standard deviation lines for all boxes, but only make the last box numbers flexible
   for (int i=0; i<limit; i++)  
   {
      datetime timei = Time[i] - (datetime) MathMod(Time[i],86400);
      dt1 = timei + dtStartTime;
      dt2 = timei + dtEndTime;
      int ib1 = iBarShift(NULL,0,dt1);
      if(Time[ib1] < dt1 && ib1>=1) { ib1--; dt1 = Time[ib1]; }
      int ib2 = iBarShift(NULL,0,dt2);
      if(Time[ib2] < dt2 && ib2>=1) { ib2--; dt2 = Time[ib2]; }
      if (dt2 != Time[i])  continue;

      int barHigh = iHighest(NULL,0,MODE_HIGH,ib1-ib2,ib2+1);
      int barLow = iLowest(NULL,0,MODE_LOW,ib1-ib2,ib2+1);
      double vHigh = High[barHigh];
      double vLow = Low[barLow];
      
      // Add ICT Standard Deviation lines (20 above and 20 below the box)
      if(ShowStandardDeviationLines)
      {
         bool isLastBox = (dt1 == lastBoxDateTime);
         drawICTStandardDeviation(dt1, dt2, vHigh, vLow, isLastBox);
      }
      else
      {
         removeStandardDeviationObjects(dt1);
      }
   }
}//end of plot_obj()

//+------------------------------------------------------------------+
void drawHorizontalLines(datetime dt1, double vHigh, double vLow)
{
   double vMid = (vHigh + vLow) / 2.0;
   
   // Calculate start and end times for horizontal lines using custom times
   datetime lineStartTime = dt1 - (datetime) MathMod(dt1,86400) + dtHorizontalLinesStartTime;
   datetime lineEndTime = dt1 - (datetime) MathMod(dt1,86400) + dtHorizontalLinesEndTime;
   
   // Draw top horizontal line
   string objnameTop = createdObjectPrefix+IntegerToString(dt1)+"_HL_T";
   if(ObjectFind(objnameTop) < 0) 
      ObjectCreate(objnameTop, OBJ_TREND, 0, lineStartTime, vHigh, lineEndTime, vHigh);
   else
   {
      ObjectSetInteger(0, objnameTop, OBJPROP_TIME1, lineStartTime);
      ObjectSetInteger(0, objnameTop, OBJPROP_TIME2, lineEndTime);
      ObjectSetDouble(0, objnameTop, OBJPROP_PRICE1, vHigh);
      ObjectSetDouble(0, objnameTop, OBJPROP_PRICE2, vHigh);
   }
   ObjectSet(objnameTop, OBJPROP_COLOR, HorizontalLinesColor);
   ObjectSet(objnameTop, OBJPROP_WIDTH, HorizontalLinesWidth);
   ObjectSet(objnameTop, OBJPROP_RAY, false);
   ObjectSet(objnameTop, OBJPROP_SELECTABLE, false);
   ObjectSet(objnameTop, OBJPROP_STYLE, STYLE_SOLID);
   
   // Draw middle horizontal line (conditionally)
   if(ShowMiddleLine)
   {
      string objnameMid = createdObjectPrefix+IntegerToString(dt1)+"_HL_M";
      if(ObjectFind(objnameMid) < 0) 
         ObjectCreate(objnameMid, OBJ_TREND, 0, lineStartTime, vMid, lineEndTime, vMid);
      else
      {
         ObjectSetInteger(0, objnameMid, OBJPROP_TIME1, lineStartTime);
         ObjectSetInteger(0, objnameMid, OBJPROP_TIME2, lineEndTime);
         ObjectSetDouble(0, objnameMid, OBJPROP_PRICE1, vMid);
         ObjectSetDouble(0, objnameMid, OBJPROP_PRICE2, vMid);
      }
      ObjectSet(objnameMid, OBJPROP_COLOR, HorizontalLinesColor);
      ObjectSet(objnameMid, OBJPROP_WIDTH, HorizontalLinesWidth);
      ObjectSet(objnameMid, OBJPROP_RAY, false);
      ObjectSet(objnameMid, OBJPROP_SELECTABLE, false);
      ObjectSet(objnameMid, OBJPROP_STYLE, STYLE_SOLID);
   }
   else
   {
      // Remove middle line if it exists and ShowMiddleLine is false
      string objnameMid = createdObjectPrefix+IntegerToString(dt1)+"_HL_M";
      if(ObjectFind(objnameMid) >= 0) ObjectDelete(objnameMid);
   }
   
   // Draw bottom horizontal line
   string objnameBottom = createdObjectPrefix+IntegerToString(dt1)+"_HL_B";
   if(ObjectFind(objnameBottom) < 0) 
      ObjectCreate(objnameBottom, OBJ_TREND, 0, lineStartTime, vLow, lineEndTime, vLow);
   else
   {
      ObjectSetInteger(0, objnameBottom, OBJPROP_TIME1, lineStartTime);
      ObjectSetInteger(0, objnameBottom, OBJPROP_TIME2, lineEndTime);
      ObjectSetDouble(0, objnameBottom, OBJPROP_PRICE1, vLow);
      ObjectSetDouble(0, objnameBottom, OBJPROP_PRICE2, vLow);
   }
   ObjectSet(objnameBottom, OBJPROP_COLOR, HorizontalLinesColor);
   ObjectSet(objnameBottom, OBJPROP_WIDTH, HorizontalLinesWidth);
   ObjectSet(objnameBottom, OBJPROP_RAY, false);
   ObjectSet(objnameBottom, OBJPROP_SELECTABLE, false);
   ObjectSet(objnameBottom, OBJPROP_STYLE, STYLE_SOLID);
   
   // Add text on horizontal lines if enabled
   if(ShowTextOnHorizontalLines)
   {
      drawHorizontalLineText(dt1, vHigh, vMid, vLow, lineStartTime, lineEndTime);
   }
   else
   {
      removeHorizontalLineText(dt1);
   }
}//end of drawHorizontalLines()

//+------------------------------------------------------------------+
void drawHorizontalLineText(datetime dt1, double vHigh, double vMid, double vLow, datetime lineStartTime, datetime lineEndTime)
{
   // Calculate text position based on user selection
   datetime textTime;
   switch(HorizontalLinesTextPosition)
   {
      case 0: // Start of line
         textTime = lineStartTime;
         break;
      case 1: // Middle of line
         textTime = lineStartTime + (lineEndTime - lineStartTime) / 2;
         break;
      case 2: // End of line
      default:
         textTime = lineEndTime;
         break;
   }
   
   // Top line text
   string textNameTop = createdObjectPrefix+IntegerToString(dt1)+"_HL_TXT_T";
   if(ObjectFind(textNameTop) < 0)
      ObjectCreate(textNameTop, OBJ_TEXT, 0, textTime, vHigh);
   else
   {
      ObjectSetInteger(0, textNameTop, OBJPROP_TIME1, textTime);
      ObjectSetDouble(0, textNameTop, OBJPROP_PRICE1, vHigh);
   }
   ObjectSetString(0, textNameTop, OBJPROP_TEXT, HorizontalLinesTopText);
   ObjectSetInteger(0, textNameTop, OBJPROP_COLOR, HorizontalLinesTextColor);
   ObjectSetInteger(0, textNameTop, OBJPROP_FONTSIZE, HorizontalLinesTextSize);
   ObjectSetInteger(0, textNameTop, OBJPROP_SELECTABLE, false);
   ObjectSetInteger(0, textNameTop, OBJPROP_BACK, false);
   
   // Middle line text (if shown)
   if(ShowMiddleLine)
   {
      string textNameMid = createdObjectPrefix+IntegerToString(dt1)+"_HL_TXT_M";
      if(ObjectFind(textNameMid) < 0)
         ObjectCreate(textNameMid, OBJ_TEXT, 0, textTime, vMid);
      else
      {
         ObjectSetInteger(0, textNameMid, OBJPROP_TIME1, textTime);
         ObjectSetDouble(0, textNameMid, OBJPROP_PRICE1, vMid);
      }
      ObjectSetString(0, textNameMid, OBJPROP_TEXT, HorizontalLinesMiddleText);
      ObjectSetInteger(0, textNameMid, OBJPROP_COLOR, HorizontalLinesTextColor);
      ObjectSetInteger(0, textNameMid, OBJPROP_FONTSIZE, HorizontalLinesTextSize);
      ObjectSetInteger(0, textNameMid, OBJPROP_SELECTABLE, false);
      ObjectSetInteger(0, textNameMid, OBJPROP_BACK, false);
   }
   
   // Bottom line text
   string textNameBottom = createdObjectPrefix+IntegerToString(dt1)+"_HL_TXT_B";
   if(ObjectFind(textNameBottom) < 0)
      ObjectCreate(textNameBottom, OBJ_TEXT, 0, textTime, vLow);
   else
   {
      ObjectSetInteger(0, textNameBottom, OBJPROP_TIME1, textTime);
      ObjectSetDouble(0, textNameBottom, OBJPROP_PRICE1, vLow);
   }
   ObjectSetString(0, textNameBottom, OBJPROP_TEXT, HorizontalLinesBottomText);
   ObjectSetInteger(0, textNameBottom, OBJPROP_COLOR, HorizontalLinesTextColor);
   ObjectSetInteger(0, textNameBottom, OBJPROP_FONTSIZE, HorizontalLinesTextSize);
   ObjectSetInteger(0, textNameBottom, OBJPROP_SELECTABLE, false);
   ObjectSetInteger(0, textNameBottom, OBJPROP_BACK, false);
}//end of drawHorizontalLineText()

//+------------------------------------------------------------------+
void removeHorizontalLineText(datetime dt1)
{
   // Remove horizontal line text objects when ShowTextOnHorizontalLines is false
   string textNameTop = createdObjectPrefix+IntegerToString(dt1)+"_HL_TXT_T";
   string textNameMid = createdObjectPrefix+IntegerToString(dt1)+"_HL_TXT_M";
   string textNameBottom = createdObjectPrefix+IntegerToString(dt1)+"_HL_TXT_B";
   
   if(ObjectFind(textNameTop) >= 0) ObjectDelete(textNameTop);
   if(ObjectFind(textNameMid) >= 0) ObjectDelete(textNameMid);
   if(ObjectFind(textNameBottom) >= 0) ObjectDelete(textNameBottom);
}//end of removeHorizontalLineText()

//+------------------------------------------------------------------+
void drawICTStandardDeviation(datetime dt1, datetime dt2, double vHigh, double vLow, bool isLastBox)
{
   double boxRange = vHigh - vLow;
   double sdValue = boxRange / 8.0; // Standard deviation value
   
   // Calculate end time for SD lines (extend until 23:00 of the same day)
   datetime dayStart = dt1 - (datetime) MathMod(dt1,86400); // Start of day (00:00)
   datetime sdEndTime = dayStart + 23 * 3600; // 23:00 of the same day
   
   // Draw 20 standard deviations above the box
   for(int i = 1; i <= 20; i++)
   {
      // Skip hidden levels
      if(isLevelHidden(i)) continue;
      
      double levelAbove = vHigh + (sdValue * i);
      string objnameAbove = createdObjectPrefix+IntegerToString(dt1)+"_SD_A"+IntegerToString(i);
      
      if(ObjectFind(objnameAbove) < 0) 
         ObjectCreate(objnameAbove, OBJ_TREND, 0, dt2, levelAbove, sdEndTime, levelAbove);
      else
      {
         ObjectSetInteger(0, objnameAbove, OBJPROP_TIME1, dt2);
         ObjectSetInteger(0, objnameAbove, OBJPROP_TIME2, sdEndTime);
         ObjectSetDouble(0, objnameAbove, OBJPROP_PRICE1, levelAbove);
         ObjectSetDouble(0, objnameAbove, OBJPROP_PRICE2, levelAbove);
      }
      
      ObjectSet(objnameAbove, OBJPROP_COLOR, ICT_SD_Color);
      ObjectSet(objnameAbove, OBJPROP_WIDTH, ICT_SD_LineWidth);
      ObjectSet(objnameAbove, OBJPROP_RAY, false);
      ObjectSet(objnameAbove, OBJPROP_SELECTABLE, false);
      ObjectSet(objnameAbove, OBJPROP_STYLE, ICT_SD_LineStyle);
      
      // Add number text for this standard deviation line
      if(ShowNumbersOnSDLines)
      {
         createNumberLabel(dt1, i, levelAbove, true, isLastBox, sdEndTime);
      }
      else
      {
         // Remove text if it exists and ShowNumbersOnSDLines is false
         string textNameAbove = createdObjectPrefix+IntegerToString(dt1)+"_SD_TXT_A"+IntegerToString(i);
         if(ObjectFind(textNameAbove) >= 0) ObjectDelete(textNameAbove);
      }
   }
   
   // Draw 20 standard deviations below the box
   for(int i = 1; i <= 20; i++)
   {
      // Skip hidden levels
      if(isLevelHidden(i)) continue;
      
      double levelBelow = vLow - (sdValue * i);
      string objnameBelow = createdObjectPrefix+IntegerToString(dt1)+"_SD_B"+IntegerToString(i);
      
      if(ObjectFind(objnameBelow) < 0) 
         ObjectCreate(objnameBelow, OBJ_TREND, 0, dt2, levelBelow, sdEndTime, levelBelow);
      else
      {
         ObjectSetInteger(0, objnameBelow, OBJPROP_TIME1, dt2);
         ObjectSetInteger(0, objnameBelow, OBJPROP_TIME2, sdEndTime);
         ObjectSetDouble(0, objnameBelow, OBJPROP_PRICE1, levelBelow);
         ObjectSetDouble(0, objnameBelow, OBJPROP_PRICE2, levelBelow);
      }
      
      ObjectSet(objnameBelow, OBJPROP_COLOR, ICT_SD_Color);
      ObjectSet(objnameBelow, OBJPROP_WIDTH, ICT_SD_LineWidth);
      ObjectSet(objnameBelow, OBJPROP_RAY, false);
      ObjectSet(objnameBelow, OBJPROP_SELECTABLE, false);
      ObjectSet(objnameBelow, OBJPROP_STYLE, ICT_SD_LineStyle);
      
      // Add number text for this standard deviation line
      if(ShowNumbersOnSDLines)
      {
         createNumberLabel(dt1, i, levelBelow, false, isLastBox, sdEndTime);
      }
      else
      {
         // Remove text if it exists and ShowNumbersOnSDLines is false
         string textNameBelow = createdObjectPrefix+IntegerToString(dt1)+"_SD_TXT_B"+IntegerToString(i);
         if(ObjectFind(textNameBelow) >= 0) ObjectDelete(textNameBelow);
      }
   }
}//end of drawICTStandardDeviation()

//+------------------------------------------------------------------+
void createNumberLabel(datetime dt1, int number, double priceLevel, bool isAbove, bool isLastBox, datetime sdEndTime)
{
   // Skip hidden levels
   if(isLevelHidden(number)) return;
   
   string prefix = isAbove ? "_SD_TXT_A" : "_SD_TXT_B";
   string textName = createdObjectPrefix+IntegerToString(dt1)+prefix+IntegerToString(number);
   
   // Calculate text position based on user selection
   datetime textTime;
   switch(NumbersTextPosition)
   {
      case 0: // Start of line
         textTime = dt1;
         break;
      case 1: // Middle of line
         textTime = dt1 + (sdEndTime - dt1) / 2;
         break;
      case 2: // End of line
      default:
         textTime = sdEndTime;
         break;
   }
   
   // For the last box, use OBJ_LABEL for horizontal flexibility but keep price-based vertical positioning
   if(isLastBox && NumbersTextPosition == 2)
   {
      // Use OBJ_LABEL for last box (flexible horizontally, fixed vertically)
      if(ObjectFind(textName) < 0)
      {
         ObjectCreate(textName, OBJ_LABEL, 0, 0, 0);
         ObjectSet(textName, OBJPROP_CORNER, CORNER_RIGHT_UPPER);
         ObjectSet(textName, OBJPROP_XDISTANCE, 10);
         ObjectSet(textName, OBJPROP_YDISTANCE, CalculateYDistance(priceLevel));
         // Store the price level in the object for vertical positioning
         ObjectSetString(0, textName, OBJPROP_TOOLTIP, DoubleToStr(priceLevel, 8));
      }
      else
      {
         // Update Y position based on current price scale to keep it fixed vertically
         ObjectSet(textName, OBJPROP_YDISTANCE, CalculateYDistance(StringToDouble(ObjectGetString(0, textName, OBJPROP_TOOLTIP))));
      }
   }
   else
   {
      // Use OBJ_TEXT for all other boxes and positions
      if(ObjectFind(textName) < 0)
      {
         ObjectCreate(textName, OBJ_TEXT, 0, textTime, priceLevel);
      }
      else
      {
         ObjectSetInteger(0, textName, OBJPROP_TIME1, textTime);
         ObjectSetDouble(0, textName, OBJPROP_PRICE1, priceLevel);
      }
   }
   
   ObjectSetString(0, textName, OBJPROP_TEXT, IntegerToString(number));
   ObjectSetInteger(0, textName, OBJPROP_COLOR, NumbersTextColor);
   ObjectSetInteger(0, textName, OBJPROP_FONTSIZE, NumbersTextSize);
   ObjectSetInteger(0, textName, OBJPROP_SELECTABLE, false);
   ObjectSetInteger(0, textName, OBJPROP_BACK, false);
   ObjectSetInteger(0, textName, OBJPROP_HIDDEN, false);
}//end of createNumberLabel()

//+------------------------------------------------------------------+
int CalculateYDistance(double priceLevel)
{
   // Calculate Y distance from top of chart for label positioning
   double chartTop = ChartGetDouble(0, CHART_PRICE_MAX);
   double chartBottom = ChartGetDouble(0, CHART_PRICE_MIN);
   double chartHeight = chartTop - chartBottom;
   
   if(chartHeight <= 0) return 20; // Avoid division by zero
   
   double pricePosition = (chartTop - priceLevel) / chartHeight;
   
   int screenHeight = (int)ChartGetInteger(0, CHART_HEIGHT_IN_PIXELS);
   return (int)(pricePosition * screenHeight) + 20; // 20 pixels offset from top
}//end of CalculateYDistance()

//+------------------------------------------------------------------+
void removeStandardDeviationObjects(datetime dt1)
{
   // Remove all standard deviation lines and text objects when ShowStandardDeviationLines is false
   for(int i = 1; i <= 20; i++)
   {
      string objnameAbove = createdObjectPrefix+IntegerToString(dt1)+"_SD_A"+IntegerToString(i);
      string objnameBelow = createdObjectPrefix+IntegerToString(dt1)+"_SD_B"+IntegerToString(i);
      string textNameAbove = createdObjectPrefix+IntegerToString(dt1)+"_SD_TXT_A"+IntegerToString(i);
      string textNameBelow = createdObjectPrefix+IntegerToString(dt1)+"_SD_TXT_B"+IntegerToString(i);
      
      if(ObjectFind(objnameAbove) >= 0) ObjectDelete(objnameAbove);
      if(ObjectFind(objnameBelow) >= 0) ObjectDelete(objnameBelow);
      if(ObjectFind(textNameAbove) >= 0) ObjectDelete(textNameAbove);
      if(ObjectFind(textNameBelow) >= 0) ObjectDelete(textNameBelow);
   }
}//end of removeStandardDeviationObjects()

//+------------------------------------------------------------------+
void del_obj(string prefix="")  
{
   if(StringLen(prefix)==0) prefix = createdObjectPrefix;
   int limit = ObjectsTotal() - 1;
   for(int k=limit; k>= 0; k--) // When deleting objects, must *decrement*.
   {
      string objname = ObjectName(k);
      if (StringFind(objname,prefix,0) == 0)  
      ObjectDelete(objname);
   }    
}//end of del_obj()

//+------------------------------------------------------------------+ 
int stringSplitBySpaces(string& output[], string s_input) export
{
   int pos, arraysize;
   ArrayResize(output, 0);
   s_input = StringTrimLeft(StringTrimRight(s_input));  // Leading/trailing spaces must be stripped off first.
   while (true) 
   {
      pos = StringFind(s_input, " ");
      arraysize = ArraySize(output);
      ArrayResize(output, arraysize + 1);
      if (pos != -1) 
      {
         output[arraysize] = StringTrimLeft(StringTrimRight(StringSubstr(s_input, 0, pos)));
         s_input = StringTrimLeft(StringTrimRight(StringSubstr(s_input, pos + 1)));
      } else {
         output[arraysize] = StringTrimLeft(StringTrimRight(s_input));
         break;
      } // if
   } // while
   return(ArraySize(output));
}//end of stringSplitBySpaces()

//+------------------------------------------------------------------+
// Splits s_input string into output array according to *multiple* single-single character delimiters.
int stringSplitByDelimiters(string& output[], string s_input, string delimiters, bool trimLeftRightSpaces=true, int finalSize=0, string defaultValue=NULL) export
{
   int p, d, len, pos, size, cycles=0;
   int countDelim = StringLen(delimiters);
   if(countDelim == 0) return(-1); // Error, no delimiter specified.  (Should I assume... " "(space), or "/" or ... ?)
   int delim[];
   ArrayResize(delim,countDelim);
   //Note: A space is not a delimiter unless it is included as a character in the delimiters
   for(p=0; p<countDelim; p++) {delim[p] = (uchar) StringGetChar(delimiters,p);} //if(delim[p]== a space, do anything different?? Force trimLeftRightSpaces=true, or ??
   
   if(trimLeftRightSpaces) s_input = StringTrimRight(s_input); // Left is always trimmed in the loop
   ArrayResize(output, 0);
   
   //arrayPrint(delim,true);

   while (true) 
   {  //find each array element
      cycles++; if(cycles>10000) return(-1);
      
      if(trimLeftRightSpaces) s_input = StringTrimLeft(s_input);
      pos=0;
      len=StringLen(s_input);
      while(true)
      { //process each character
         cycles++; if(cycles>10000) return(-1);
         ushort asciiCode = StringGetCharacter(s_input,pos);
         bool delimFound=false;
         for(d=0; d<countDelim; d++)
         {
            if(pos>=len || asciiCode == delim[d])
            {
               //found a delim or EOS at pos
               delimFound=true;
               break;
            }
         }
         if(delimFound) break;
         pos++;
      }
      size = ArraySize(output);
      ArrayResize(output, size + 1);
      //if (pos == 0) output[size] = ""; //empty element 
      if (pos == 0) output[size] = defaultValue; //empty element
      else 
      {
         output[size] = StringSubstr(s_input, 0, pos);
         if(trimLeftRightSpaces) output[size] = StringTrimLeft(StringTrimRight(output[size]));
      }
      if(pos>=len) break;
      s_input = StringSubstr(s_input, pos + 1); // was len_token which is always 1
      
      //Alert("DEBUG (pre)size=",size,"  output[size]='",output[size],"'","  Remaining s_input='",s_input,"'  d=",d,"  pos=",pos);
   }
   
   int arSize = ArraySize(output);
   if(finalSize>arSize)
   {
      ArrayResize(output,finalSize);
      for(int k=arSize; k<finalSize; k++) {output[k] = defaultValue;}
      arSize = finalSize;
   }
   return(arSize);

   return(ArraySize(output));       
}//end of stringSplitByDelimiters()

//+------------------------------------------------------------------+
string stringReplaceEveryMatch(string str, string toFind, string toReplace, bool ignoreCaseOf_toFind=false) export {
   //Updated v3_15 to support ignoreCaseOf_toFind
   int len_toReplace = StringLen(toReplace);
   int len_toFind = StringLen(toFind);
   //int len = StringLen(toFind);
   int pos = 0;
   string leftPart, rightPart, result = str, TOFIND = toFind;
   if(ignoreCaseOf_toFind) StringToUpper(TOFIND);

   if (len_toFind == 0 || (!ignoreCaseOf_toFind && toFind == toReplace)) return (result); // Cannot find "", and replacing same-with-same is a waste of time.
   while (true) {
       // Careful, (pos or result) must change each loop or it's an infinite loop.
       if(ignoreCaseOf_toFind) pos = StringFind(stringToUC(result), TOFIND, pos);
       else pos = StringFind(result, toFind, pos);
       if (pos == -1) {
           break;
       }
       if (pos == 0) {
           leftPart = "";
       } else {
           leftPart = StringSubstr(result, 0, pos);
       }
       rightPart = StringSubstr(result, pos+len_toFind); 
       result = StringConcatenate(leftPart,toReplace,rightPart);
       pos = pos + len_toReplace;
   }    
   return (result);
}//end of stringReplaceEveryMatch()
//+------------------------------------------------------------------+
string stringToUC(string str) export {
   // Convert str to upper-case
   int lS = 97, lE = 122, uS = 65, uE = 90, diff = lS - uS;
   for (int i = 0; i < StringLen(str); i++) {
       int code = StringGetChar(str, i);
       if (code >= lS && code <= lE) {
           code -= diff;
           str = StringSetChar(str, i, (ushort) code);
       }
   }
   return (str);
}//end of stringToUC()
//+------------------------------------------------------------------+
bool IsNewBar() export // DEV to add: string symbol=_Symbol, int period=_Period)
{
   static datetime lastbar; // DEV Q: Would a single lastbar var work with multiple different calls and/or arguments to function? If no, how? Pass the &lastbar?
   datetime curbar = (datetime)SeriesInfoInteger(_Symbol,_Period,SERIES_LASTBAR_DATE);
   if(lastbar != curbar)
   {
      lastbar = curbar;
      return true;
   }
   return false;
}//end of IsNewBar()

//+------------------------------------------------------------------+
color colorToRGBflip(color c_mycolor) export { return( (color) (~c_mycolor & 0xFFFFFF) ); }
color colorToVisibleRGBflip(color c_mycolor) export 
{
   if(isBackgroundLight())
   {
      return( (color) (~c_mycolor & 0xBBBBBB) );
   }
   else
   {
      return( (color) ((~c_mycolor & 0xFFFFFF) | 0x444444) );
   }
}//end of colorToVisibleRGBflip()
//+------------------------------------------------------------------+
int   colorToRGBsum(color c_mycolor) export { return( (c_mycolor & 0xFF) + ((c_mycolor>>8) & 0xFF) + ((c_mycolor>>16) & 0xFF) ); }
bool  isBackgroundLight() export { if(colorToRGBsum( (int) ChartGetInteger(0,CHART_COLOR_BACKGROUND)) >382 ) return(true); else return(false); }
//+------------------------------------------------------------------+
bool stringIsBlank(string str, bool do_StringTrim_LR=true) export
{
   if(do_StringTrim_LR) str = StringTrimLeft(StringTrimRight(str));
   if(StringGetChar(str,0) > 0) return(false);
   return(true);
}//end of stringIsBlank()
//+------------------------------------------------------------------+